# getmode <- function(v) {
#   uniqv <- unique(v)
#   uniqv[which.max(tabulate(match(v, uniqv)))]
# }


#' Function used in [robustfun2()]
#'
#' @param x
#'
#' @return A real number.
#' @noRd
robustfun <- function(x) {
  if (x >= 1) {
    y <- log(2)
  }
  if ((x >= 0) && (x < 1)) {
    y <- -log(1 - x + x^2 / 2)
  }
  if ((x <= 0) && (x > -1)) {
    y <- log(1 + x + x^2 / 2)
  }
  if (x <= -1) {
    y <- -log(2)
  }
  return(y)
}


#' Function used in [get_lr_var()]
#'
#' @param x An array of real numbers.
#' @param c A parameter in M-estimation zero function.
#' @param alpha A positive number.
#'
#' @return A real number.
#' @noRd
robustfun2 <- function(x, c, alpha) {
  y <- x
  for (i in (1:length(x)))
  {
    y[i] <- alpha^(-1) * robustfun(alpha * (x[i] - c))
  }
  ysum <- sum(y) / length(x)
  return(ysum)
}


#' Fucuntion used in [get_lr_var()]
#'
#' @param x An array of real numbers.
#' @param size A denominator.
#'
#' @return A real number.
#' @noRd
varest <- function(x, size) {
  sum(x) / size
}


#' Fucuntion used in [get_GS_MAinf()]
#'
#' @param s A loop counter.
#' @param i A loop counter.
#' @param n Number of time series observations.
#' @param p Number of individuals.
#' @param b Bandwith parameter \eqn{b = window_size/n}.
#' @param z_MAinf Gaussian random vector Z.
#'
#' @return A real number.
#' @noRd
GS_MAinf_s_i <- function(s, i, n, p, b, z_MAinf) {
  sum(z_MAinf[((s - 1) * p + 1):(s * p), i]) / (b * n) + 2 * p / (b * n)
}

#' Fucuntion used in [get_GS_MAinf()]
#'
#' @param s A loop counter.
#' @param i A loop counter.
#' @param r A loop counter.
#' @param n Number of time series observations.
#' @param p Number of individuals.
#' @param b Bandwith parameter \eqn{b = window_size/n}.
#' @param nbd_info A list containing the neighbourhood information. See [ts_hdchange()].
#' @param nbd_size Size of each neighborhood.
#' @param z_MAinf Gaussian random vector Z.
#'
#' @return A real number.
#' @noRd
GS_MAinf_s_i_r <- function(s, i, r, n, p, b, nbd_info, nbd_size, z_MAinf) {
  nbd_factorK <- (p / nbd_size[r])^(1 / 4)
  row_take <- c(((s - 1) * p + 1):(s * p))
  sum(nbd_factorK^2 * z_MAinf[row_take[nbd_info[[r]]], i]) / (b * n) + 2 *
    nbd_size[r] * nbd_factorK^2 / (b * n)
}


#' Fucuntion used in [get_V_l2_MAinf()]
#'
#' @param i A loop counter.
#' @param j A loop counter.
#' @param n Number of time series observations.
#' @param p Number of individuals.
#' @param b Bandwith parameter \eqn{b = window_size/n}.
#' @param data The data matrix.
#' @param var_MAinf The diagonal matrix whose elements corresponds to the ones
#' of the longrun covariance matrix.
#' @param weight \eqn{=1/(b*n)}.
#'
#' @return A real number.
#' @noRd
V_MAinf_i_j <- function(i, j, n, p, b, data, var_MAinf, weight) {
  mu_hat_left <- weight * sum(data[j, i:(i + b * n - 1)])
  mu_hat_right <- weight * sum(data[j, (i + b * n):(i + 2 * b * n - 1)])
  (mu_hat_left - mu_hat_right) / sqrt(var_MAinf[j])
}


#' Check the validity of the neighbourhood specification
#'
#' @param nbd_info A list containing the neighbourhood information. See [ts_hdchange()].
#'
#' @return No return value. Show an error message if nbd_info is invalid.
#' @export
#'
#' @examples
#' nbd_info <- list(c(1:10),c(8:20))
#' check_nbd <- check_nbd(nbd_info)
#'
#'
check_nbd <- function(nbd_info) {

  if (!is.list(nbd_info)) {
    stop("Argument nbd_info should be a list")
  }
  S_h <- length(nbd_info)

  if (S_h == 0) {
    stop("Length of neighbourhood should be larger than zero")
  }else{
    cat("The neibourhood information is valid.")
    return(invisible(NULL))
   }

}


#' Construct an S3 class 'no_nbd' or 'nbd' for change-point estimation
#'
#' @param hdobj An S3 object of class 'no_nbd' or 'nbd' generated by [ts_hdchange()].
#' @param test_stats A list containing the test statistics generated by [get_teststats()].
#' @param threshold The threshold in break estimation.
#' @param stat_all An array of test statistics generated by [get_V_l2_MAinf()].
#' @param critical_values An array of quantiles for critical values.
#'
#' @return An S3 object of class 'no_nbd' or 'nbd' used as the argument of [get_breaks()].
#' @export
#'
#' @examples
#' # generate data
#' data_no_nbd <- sim_hdchange_no_nbd(n = 200,
#' p = 30,
#' S = 30,
#' tau = c(40, 100, 160),
#' dist_info =
#'   list(dist = "normal", dependence = "MA_inf", param = 1),
#' jump_max = c(2, 2, 1.5))
#'
#' # construct no_nbd object
#' ts_no_nbd <- ts_hdchange(data_no_nbd,
#' window_size = 30,
#' m = 8,
#' h = 1,
#' N_rep = 999,
#' alpha = 1e-5,
#' quantiles = c(0.01, 0.05, 0.1))
#'
#' teststats <- get_teststats(ts_no_nbd)
#' V_12_MAinf <- get_V_l2_MAinf(ts_no_nbd)
#'
#' estobj <- est_hdchange(hdobj = ts_no_nbd, test_stats = teststats$stat_max,
#' threshold = 1e-5, stat_all = V_12_MAinf, critical_values = c(0.01, 0.05, 0.1))
#'
#'
est_hdchange <- function(hdobj,
                         test_stats,
                         threshold,
                         stat_all,
                         critical_values) {
  arg_list<-list(
    hdobj,
    test_stats,
    threshold,
    stat_all,
    critical_values
  )
  if (is.null(hdobj$nbd_info)) {
    ret_list <- structure(arg_list,
                          class = c("no_nbd")
    )
  } else {
    ret_list <- structure(arg_list,
                          class = c("nbd")
    )
  }

  names(ret_list) <-
    c(
      "hdobj",
      "test_stats",
      "threshold",
      "stat_all",
      "critical_values"
    )

  return(ret_list)
}
