% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ext_di.R
\name{ext_di}
\alias{ext_di}
\title{Extinction Risk Estimation for a Density-Independent Model}
\usage{
ext_di(
  dat,
  ne = 1,
  th = 100,
  alpha = 0.05,
  unit = "years",
  qq_plot = FALSE,
  formatted = TRUE,
  digits = getOption("extr.digits", 5L)
)
}
\arguments{
\item{dat}{Data frame with two numeric columns: time (strictly increasing)
and population size. Column names are not restricted.}

\item{ne}{Numeric. Extinction threshold \eqn{n_e \ge 1}. Default is 1.}

\item{th}{Numeric. Time horizon \eqn{t^{\ast} > 0}. Default is 100.}

\item{alpha}{Numeric. Significance level \eqn{\alpha \in (0,1)}.
Default is 0.05.}

\item{unit}{Character. Unit of time (e.g., "years", "days", "generations").
Default is "years".}

\item{qq_plot}{Logical. If \code{TRUE}, draws a QQ-plot of standardized
increments to check model assumptions. Default is \code{FALSE}.}

\item{formatted}{Logical. If \code{TRUE}, returns an \code{"ext_di"} object;
otherwise returns a raw list. Default is \code{TRUE}.}

\item{digits}{Integer. Preferred significant digits for printing. Affects
display only. Default is \code{getOption("extr.digits", 5)}.}
}
\value{
A list (class \code{"ext_di"} if \code{formatted=TRUE}) with:
\itemize{
\item \code{Growth.rate}, \code{Variance}, \code{Unbiased.variance};
\item \code{AIC};
\item \code{Extinction.probability} with confidence limits;
\item data summary (\code{nq}, \code{xd}, \code{sample.size});
\item input parameters (\code{unit}, \code{ne}, \code{th}, \code{alpha}).
}
}
\description{
Estimates demographic parameters and extinction probability under a
density-independent (drifted Wiener) model. From a time series of
population sizes, it computes MLEs of growth rate and environmental
variance, then evaluates extinction risk over a horizon \eqn{t^{\ast}}.
Confidence intervals are constructed by the \eqn{w}-\eqn{z} method,
which achieve near-nominal coverage across the full parameter space.
}
\details{
Population dynamics follow \deqn{dX=\mu\,dt+\sigma\,dW,} where
\eqn{X(t)=\log N(t)}, \eqn{\mu} is the growth rate, \eqn{\sigma^2} the
environmental variance, and \eqn{W} a Wiener process. Extinction risk is
\deqn{G=\Pr[T\le t^{\ast}\mid N(0)=n_0,n_e,\mu,\sigma],} the probability the
population falls below \eqn{n_e} within \eqn{t^{\ast}}. Irregular intervals
are allowed.

The function:
\enumerate{
\item estimates \eqn{\mu} and \eqn{\sigma^2} (Dennis et al., 1991),
\item computes extinction probability \eqn{G(w,z)}
(Lande and Orzack, 1988),
\item constructs confidence intervals for \eqn{G} using the
\eqn{w}-\eqn{z} method (Hakoyama, 2025).
}

Numerical range. Probabilities are evaluated on \eqn{G}, \eqn{\log G}, and
\eqn{\log(1-G)} scales. The log-scale removes the \eqn{\approx
4.94\times10^{-324}} lower bound of linear doubles and extends the safe
range down to \code{exp(-DBL_MAX)} (kept symbolically), avoiding
underflow/cancellation.
}
\examples{
# Example from Dennis et al. (1991), Yellowstone grizzly bears
dat <- data.frame(Time = 1959:1987,
Population = c(44, 47, 46, 44, 46, 45, 46, 40, 39, 39, 42, 44, 41, 40,
33, 36, 34, 39, 35, 34, 38, 36, 37, 41, 39, 51, 47, 57, 47))

# Probability of decline to 1 individual within 100 years
ext_di(dat, th = 100)

# Probability of decline to 10 individuals within 100 years
ext_di(dat, th = 100, ne = 10)

# With QQ-plot
ext_di(dat, th = 100, qq_plot = TRUE)

# Change digits
ext_di(dat, th = 100, ne = 10, digits = 9)

}
\references{
Lande, R. and Orzack, S.H. (1988) Extinction dynamics of age-structured
populations in a fluctuating environment. \emph{Proceedings of the National
Academy of Sciences}, 85(19), 7418–7421.

Dennis, B., Munholland, P.L., and Scott, J.M. (1991) Estimation of growth
and extinction parameters for endangered species. \emph{Ecological Monographs},
61, 115–143.

Hakoyama, H. (2025) Confidence intervals for extinction risk: validating
population viability analysis with limited data.
Preprint, doi:10.48550/arXiv.2509.09965
}
\seealso{
\code{\link{statistics_di}},
\code{\link{extinction_probability_di}},
\code{\link{confidence_interval_wz_di}}, \code{\link{print.ext_di}}
}
\author{
Hiroshi Hakoyama, \email{hiroshi.hakoyama@gmail.com}
}
\keyword{methods}
\keyword{models}
\keyword{survival}
\keyword{time-series}
